/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2011-2018 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Description

\*---------------------------------------------------------------------------*/

#include "argList.H"
#include "IOmanip.H"
#include "ODESystem.H"
#include "ODESolver.H"

using namespace Foam;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

    float Cpmax = 51410 ;
    float Cnmax = 31833 ;

    float Dpr = 1.0e-14;
    float Rp = 8.5e-6;
    float Lp = 70e-6;
    float kpr = 6.6667e-11;
    float Sp = 1.1167 ;
    float Ep = 29000 ;
    float Eap = 58000 ;

    float Ce = 1000;
    float F = 96487;
    float Tamb = 273+35;
    float ho = 1 ;  //Assumed not given
    float rho = 1626;
    float Vol = 0.199*0.08499*0.002;
    float Cap = 750;
    float Area = 0.085 ;
    float R = 8.314;
    float Ic = -1.656 ;

    float Dnr = 3.9e-14;
    float Rn = 12.5e-6;
    float Ln = 73.5e-6;
    float knr = 1.764e-11;
    float Sn = 0.7824 ;
    float En = 35000 ;
    float Ean = 20000 ;

    float delrp = Rp/10;
    float delrn = Rn/10;


    float Dp;
    float Dn;
    float fp;
    float fn;
    float kp;
    float kn;
    float k;
    float k2;




class testODE:

    public ODESystem
    {


    public:

    testODE()
    {}

    label nEqns() const
    {
        return 18;
    }


    void derivatives
    (
        const scalar x,
        const scalarField& y,
        scalarField& dydx
    ) const
    {

        dydx[0] = 2*k*(y[1]-y[0] ) ;

        for(int i = 1 ; i < 8 ; i++) dydx[i] = k*( (1/(i+1))*( y[i+1] - y[i-1] ) + y[i+1] - 2*y[i] + y[i-1] );

        dydx[8] = k*( (1/9)*( +fp/1.5 + (4/3)*y[8] - (4/3)*y[7] ) + ( fp/1.5 - (2/3)*y[8] +(2/3)*y[7] ) );



        dydx[9]  = 2*k2*(  y[10] - y[9] );

        for(int i = 10 ; i < 17 ; i++) dydx[i] = k2*( (1/(i-8))*( y[i+1] - y[i-1] ) + y[i+1] - 2*y[i] + y[i-1] );

        dydx[17] = k2*( (1/9)*( -fn/1.5 + (4/3)*y[17] - (4/3)*y[16] ) + ( -fn/1.5 - (2/3)*y[17] +(2/3)*y[16] ) );


    }

    void jacobian
    (
        const scalar x,
        const scalarField& y,
        scalarField& dfdx,
        scalarSquareMatrix& dfdy
    ) const
    {

        for (int j=0; j<17; j++) dfdx[j] = 0.0;

        for (int j=0; j<18; j++)
        {
          for(k=0; k<18; k++) dfdy(j, k) = 0.0;
        }


        dfdy(0, 0) = -2*k;
        dfdy(0, 1) = 2*k;

        dfdy(1, 0) = 0.5*k;
        dfdy(1, 1) = -2*k;
        dfdy(1, 2) = 1.5*k;

        dfdy(2, 1) = (2/3)*k;
        dfdy(2, 2) = -2*k;
        dfdy(2, 3) = (4/3)*k;

        dfdy(3, 2) = 0.75*k;
        dfdy(3, 3) = -2*k;
        dfdy(3, 4) = 1.25*k;

        dfdy(4, 3) = 0.8*k;
        dfdy(4, 4) = -2*k;
        dfdy(4, 5) = 1.2*k;

        dfdy(5, 4) = (5/6)*k;
        dfdy(5, 5) = -2*k;
        dfdy(5, 6) = (7/6)*k;

        dfdy(6, 5) = (6/7)*k;
        dfdy(6, 6) = -2*k;
        dfdy(6, 7) = (8/7)*k;

        dfdy(7, 6) = 0.875*k;
        dfdy(7, 7) = -2*k;
        dfdy(7, 8) = 1.125*k;

        dfdy(8, 7) = (14/27)*k;
        dfdy(8, 8) = -(14/27)*k;



        dfdy(9, 9) = -2*k2;
        dfdy(9, 10) = 2*k2;

        dfdy(10, 9) = 0.5*k2;
        dfdy(10, 10) = -2*k2;
        dfdy(10, 11) = 1.5*k2;

        dfdy(11, 10) = (2/3)*k2;
        dfdy(11, 11) = -(2)*k2;
        dfdy(11, 12) = (4/3)*k2;

        dfdy(12, 11) = 0.75*k2;
        dfdy(12, 12) = -2*k2;
        dfdy(12, 13) = (1.25)*k2;

        dfdy(13, 12) = 0.8*k2;
        dfdy(13, 13) = -2*k2;
        dfdy(13, 14) = 1.2*k2;

        dfdy(14, 13) = (5/6)*k2;
        dfdy(14, 14) = -(2)*k2;
        dfdy(14, 15) = (7/6)*k2;

        dfdy(15, 14) = (6/7)*k2;
        dfdy(15, 15) = (-2)*k2;
        dfdy(15, 16) = (8/7)*k2;

        dfdy(16, 15) = 0.875*k2;
        dfdy(16, 16) = -2*k2;
        dfdy(16, 17) = 1.125*k2;

        dfdy(17, 16) = (14/27)*k2;
        dfdy(17, 17) = -(14/27)*k2;

    }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
// Main program:
float UpT;
float UnT;
float etp;
float etn;

class tempODE
:
    public ODESystem
{

public:

    tempODE()
    {}

    label nEqns() const
    {
        return 1;
    }

    void derivatives
    (
        const scalar x,
        const scalarField& y,
        scalarField& dydx
    ) const
    {
        dydx[0] = (Ic*y[0]*(UpT-UnT) + Ic*(etp-etn + Ic*(0.0162 + 0.0162*(y[0]-Tamb) ) )  - ho*Area*(y[0]-Tamb) )/(rho*Vol*Cap) ;
    }

    void jacobian
    (
        const scalar x,
        const scalarField& y,
        scalarField& dfdx,
        scalarSquareMatrix& dfdy
    ) const
    {
        dfdx[0] = 0.0;
        dfdy(0, 0) = (Ic*(UpT-UnT) + Ic*Ic*0.0162 - ho*Area )/(rho*Vol*Cap);
    }
};

float ThermalFunction(float t, float xy[])
{

  tempODE ode2;
  dictionary dict2;
  dict2.add("solver", "RKF45");
  autoPtr<ODESolver> odeSolver2 = ODESolver::New(ode2, dict2);


  ::UpT = ( -0.19952 + 0.92837*xy[0]  -1.36455*pow(xy[0],2) + 0.61154*pow(xy[0],3))/( 1 - 5.66148*xy[0] + 11.47636*pow(xy[0],2) - 9.82431*pow(xy[0],3) + 3.04876*pow(xy[0],4) )  ;
  ::UnT = ( 0.00527 + 3.29927*xy[1] - 91.79326*pow(xy[1],2) + 1004.91101*pow(xy[1],3) - 5812.27813*pow(xy[1],4) + 19329.75490*pow(xy[1],5) - 37147.89470*pow(xy[1],6) + 38379.18127*pow(xy[1],7) - 16515.05308*pow(xy[1],8))/(1 - 48.09287*xy[1] + 1017.23480*pow(xy[1],2) - 10481.80419*pow(xy[1],3) + 59431.30001*pow(xy[1],4) - 195881.64880*pow(xy[1],5) + 374577.31520*pow(xy[1],6) - 385821.16070*pow(xy[1],7) + 165705.85970*pow(xy[1],8) ) ;


  float mp = Ic/(F*Sp*kp*Cpmax*sqrt(Ce*xy[0]*(1-xy[0]) ) );
  ::etp = 2*(R/F)*log(float( (sqrt(float( ( pow(mp,2) + 4) ) )+mp)*0.5 ) ) ;


  float mn = float(Ic/(F*Sn*kn*Cnmax*sqrt(Ce*xy[1]*(1-xy[1]) ) ) );
  ::etn = 2*(R/F)*log(float( (sqrt(float( ( pow(mn,2) + 4) ) )+mn)*0.5 ) );

  scalar xStart = 0.0;
  scalarField yStart(ode2.nEqns());
  yStart[0] = Tamb;

  scalarField dyStart(ode2.nEqns());
  ode2.derivatives(xStart, yStart, dyStart);

  scalarField y(yStart);
  scalar dxEst = 0.5;
  scalar x = t;
  scalar xEnd = t + 1 ;

  odeSolver2->solve(x, xEnd, y, dxEst);

return float(y[0]);

}

void BatteryFunction(int t, float xy[])
{

  testODE ode;
  dictionary dict;
  dict.add("solver", "RKF45");
  autoPtr<ODESolver> odeSolver = ODESolver::New(ode, dict);

  scalar xStart = 0.0;
  scalarField yStart(ode.nEqns());
  for (label i=0; i<9; i++)  yStart[i] = Cpmax*0.4952;
  for (label i=9; i<19; i++)  yStart[i] = Cnmax*0.7522;

  scalarField dyStart(ode.nEqns());
  ode.derivatives(xStart, yStart, dyStart);

  scalarField y(yStart);
  scalar dxEst = 0.5;
  scalar x = xStart;
  scalar xEnd = t ;

  odeSolver->solve(x, xEnd, y, dxEst);
  float xps= float( +fp/1.5 + (4/3)*y[8] -(1/3)*y[7])/Cpmax;
  float xns= float( -fn/1.5 + (4/3)*y[17] -(1/3)*y[16])/Cnmax;

  xy[0]=xps;
  xy[1]=xns;

}

float PotentialFunction(float xy[], float Res, float Tx)
{
  float Un = float( 0.13966 + 0.68920*exp(float(-49.20361*xy[1])) + 0.41903*exp(float(-254.40067*xy[1])) -exp(float(49.97886*xy[1] - 43.37888)) -0.028221*atan(float(22.52300*xy[1] - 3.65328)) -0.01308*atan(float(28.34801*xy[1] - 13.43960)) ) ;

  float Up = float( 4.04596 + exp(float(-42.30027*xy[0] + 16.56714)) - 0.04880*atan(float( 50.01833*xy[0] - 26.48897)) - 0.05447*atan(float(18.99678*xy[0] - 12.32362)) - exp(float(78.24095*xy[0] - 78.68074)) );

  float mp = Ic/(F*Sp*kp*Cpmax*sqrt(Ce*xy[0]*(1-xy[0]) ) );

  float etap = 2*R*(Tx/F)*log(float( (sqrt(float( ( pow(mp,2) + 4) ) ) + mp )*0.5 ) ) ;

  float mn = float(Ic/(F*Sn*kn*Cnmax*sqrt(Ce*xy[1]*(1-xy[1]) ) ) );

  float etan = 2*R*(Tx/F)*log(float( (sqrt(float( ( pow(mn ,2) + 4) ) ) + mn )*0.5 ) );

  float V = Up - Un + etap + etan + Ic*Res;

  return V;
}

int main(int argc, char *argv[])
{
    int max_step = 4000;
    float Tx = Tamb;
    float Ts[max_step];
    float Volt[max_step];
    ::Dn = Dnr;
    ::Dp = Dpr;
    ::kp = kpr;
    ::kn = knr;

    float Res;

    for (label i=0; i<max_step; i++)
    {


        Info << "Time :"<< i+1 << endl;
        float xy[2];

        BatteryFunction(i, xy);

        float xps = xy[0];
        float xns = xy[1];
        Info << "the conc " << xps << " and " << xns <<endl;

        Tx = ThermalFunction(i,xy);

        Info << "The Temperature is now " << Tx <<endl;
        Ts[i] = Tx;

        ::Dn = Dnr*exp((En/R)*((1/Tx) - (1/Tamb) ));
        ::Dp = Dpr*exp((Ep/R)*((1/Tx) - (1/Tamb)));
        ::fp = -(Ic*delrp)/(Dp*F*Sp);
        ::fn = -(Ic*delrn)/(Dn*F*Sn);

        ::k = Dp/pow(delrp,2);
        ::k2 = Dn/pow(delrn,2);

        ::kn = knr*exp((Ean/R)*((1/Tx) - (1/Tamb) ));
        ::kp = kpr*exp((Eap/R)*((1/Tx) - (1/Tamb)));
        Res = 0.0162 + 0.0162*(Tx-Tamb);

        Info << "Diffusivity at p is : " << Dp << " at n is : " << Dn <<endl;
        Info << "Rate Constant at p is : " << kp << " at n is : " << kn <<endl;

        Volt[i] = PotentialFunction(xy,Res,Tx);



    }

for (label i=0; i<max_step; i++)  Info << Volt[i] <<endl;
Info <<" The temperature is "<< endl ;
for (label i=0; i<max_step; i++)  Info << Ts[i] <<endl;


    return 0;



}


// ************************************************************************* //
